﻿/*
 i-net software provides programming examples for illustration only, without warranty
 either expressed or implied, including, but not limited to, the implied warranties
 of merchantability and/or fitness for a particular purpose. This programming example
 assumes that you are familiar with the programming language being demonstrated and
 the tools used to create and debug procedures. i-net software support professionals
 can help explain the functionality of a particular procedure, but they will not modify
 these examples to provide added functionality or construct procedures to meet your
 specific needs.
  
 © i-net software 1998-2013

*/
using System;
using System.Net;
using System.Windows.Forms;

using Inet.Viewer.Data;
using Inet.Viewer.WinForms;

namespace Inet.Viewer
{
    /// <summary>
    /// This Form is an example on how to use the ReportViewer. The textfield and the buttons at the bottom of this form do not belong to the actuall ReportViewer,
    /// but does help to use it for test purposes and demonstrates how it could be used.
    /// 
    /// <see cref="ShowReport()"> Have a look at this method to see how the ReportViewer is used and how a ReportView is created and added</see>
    /// </summary>
    public partial class ReportViewerExample : Form
    {
        private URLRenderData.IAuthenticator authenticator;
        private CookieContainer cookieContainer = new CookieContainer();
        
        /// <summary>
        /// Constructor
        /// </summary>
        public ReportViewerExample()
        {
            InitializeComponent();
            tableLayoutPanel1.BorderStyle = BorderStyle.FixedSingle;
            authenticator = new AuthenticationDialog(this);
            this.txtFile.Text = Properties.Settings.Default.lastURL;
        }

        /// <summary>
        /// Main methods that shows this ReportViewerExample
        /// </summary>
        /// <param name="args"></param>
        [STAThread]
        public static void Main(string[] args)
        {
            Application.EnableVisualStyles();
            ReportViewerExample form = new ReportViewerExample();
            form.ShowDialog();
        }

        /// <summary>
        /// Invoked when clicked on Show Button.
        /// This method checks whether the location is a url or a file location
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void btnShow_Click(object sender, EventArgs e)
        {
            ShowReport();
        }

        /// <summary>
        /// Gets the location of the report from text input field, checks which type of location it is
        /// and adds this as a ReportView. 
        /// </summary>
        private void ShowReport()
        {
            // Create a connection with UrlRenderData with the text of the textfield
            URLRenderData urlData = new URLRenderData(txtFile.Text);
            urlData.Authenticator = authenticator;
            urlData.CookieContainer = cookieContainer;

            // Create a new ReportView for this ReportData and add it to the viewer
            IReportView view = reportViewer1.AddNewReportView(urlData);

            reportViewer1.ShowToolbar = true;

            // Remember the URL for the next viewer start:
            Properties.Settings.Default.lastURL = this.txtFile.Text;
            Properties.Settings.Default.Save();
        }

        /// <summary>
        /// Show the report with pressing "Enter"
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void txtFile_KeyDown(object sender, KeyEventArgs e)
        {
            if (e.KeyCode == Keys.Enter)
            {
                ShowReport();
            }
        }

        /// <inheritdoc/>
        protected override bool ProcessCmdKey(ref Message message, Keys keys)
        {
            switch (keys)
            {
                case Keys.Control | Keys.F:
                    reportViewer1.SearchVisible = !reportViewer1.SearchVisible;
                    break;
                case Keys.Control | Keys.P:
                    if (reportViewer1.CurrentReportView != null)
                    {
                        reportViewer1.CurrentReportView.Print();
                    }
                    break;
                case Keys.Control | Keys.E:
                    if (reportViewer1.CurrentReportView != null)
                    {
                        reportViewer1.CurrentReportView.OpenExportDialog();
                    }
                    break;
                case Keys.Control | Keys.Subtract:
                    if (reportViewer1.CurrentReportView != null)
                    {
                        reportViewer1.ToolBar.ZoomOut();
                    }
                    break;
                case Keys.Control | Keys.Add:
                    if (reportViewer1.CurrentReportView != null)
                    {
                        reportViewer1.ToolBar.ZoomIn();
                    }
                    break;
                case Keys.F5:
                    if (reportViewer1.CurrentReportView != null)
                    {
                        reportViewer1.CurrentReportView.DataRefresh();
                    }
                    break;
            }
            return base.ProcessCmdKey(ref message, keys);
        }

        /// <summary>
        /// Called when the window was closed. It is recommended to invoke CloseAllReportViews() here
        /// for cleaning up the report viewer, i.e. to terminate any running background threads.
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void ReportViewerExample_FormClosed(object sender, FormClosedEventArgs e)
        {
            reportViewer1.CloseAllReportViews();
        }
    }
}
